/*
$Revision: 92 $
$Author: stephen $
$Date: 2017-09-10 17:54:15 -0400 (Sun, 10 Sep 2017) $
*/

/*
 * Copyright © Stephen Jupe
 *
 * Licensed under the terms of the GNU General Public License (GPL) version 3.
 * A copy of the license can be obtained from the Free Software Foundation 
 * at www.gnu.org.
 */

package jupe.stephen;

import java.awt.Color;
import java.awt.Container;
import java.awt.Frame;
import java.awt.GridLayout;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.ItemListener;
import java.awt.event.ItemEvent;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Properties;
import java.util.Random;
import java.util.StringTokenizer;
import java.util.Date;
import java.text.DateFormat;
import java.text.NumberFormat;
import javax.swing.BoxLayout;
import javax.swing.Timer;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JMenuBar;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JLabel;
import javax.swing.JButton;
import javax.swing.JToggleButton;
import javax.swing.JComboBox;
import javax.swing.JSeparator;
import javax.swing.JTextField;
import javax.swing.JOptionPane;
import javax.swing.JDialog;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.SwingConstants;
import javax.swing.WindowConstants;


/**
 * [Add javadoc comments here]
 * @author  Stephen Jupe
 */
public class TimesTableJupe
{
	private Properties properties;
	private String[] playerNames;
	private String playerName;
	private int settingsTop;
	private int settingsLeft;
	private int settingsHeight;
	private int settingsWidth;
	private int resultsTop;
	private int resultsLeft;
	private int resultsHeight;
	private int resultsWidth;
	private int mainTop;
	private int mainLeft;
	private int mainHeight;
	private int mainWidth;
	private int gridSize;
	private int errorCount;
	private boolean startState;


	/**
	 * This is the entry point into the program.
	 */
	public static void main(String[] args)
	{
		TimesTableJupe program = new TimesTableJupe();
		program.run();
		//System.out.println("This is the end of the 'main()' method.");  //debug
	}


	/**
	 * No args constructor.
	 */
	public TimesTableJupe()
	{
		errorCount = 0;
		startState = true;

		// create a Properties object and populate it
		loadProperties();

		// set variables for main window location and size
		try
		{
			settingsTop = Integer.parseInt(properties.getProperty("settingstop", "0"));
			settingsLeft = Integer.parseInt(properties.getProperty("settingsleft", "0"));
			settingsHeight = Integer.parseInt(properties.getProperty("settingsheight", "120"));
			settingsWidth = Integer.parseInt(properties.getProperty("settingswidth", "300"));
			resultsTop = Integer.parseInt(properties.getProperty("resultstop", "0"));
			resultsLeft = Integer.parseInt(properties.getProperty("resultsleft", "0"));
			resultsHeight = Integer.parseInt(properties.getProperty("resultsheight", "120"));
			resultsWidth = Integer.parseInt(properties.getProperty("resultswidth", "300"));
			mainTop = Integer.parseInt(properties.getProperty("maintop", "0"));
			mainLeft = Integer.parseInt(properties.getProperty("mainleft", "0"));
			mainHeight = Integer.parseInt(properties.getProperty("mainheight", "520"));
			mainWidth = Integer.parseInt(properties.getProperty("mainwidth", "520"));
		}
		catch(Exception e)
		{
			settingsTop = 0;
			settingsLeft = 0;
			settingsHeight = 120;
			settingsWidth = 300;
			resultsTop = 0;
			resultsLeft = 0;
			resultsHeight = 120;
			resultsWidth = 300;
			mainTop = 0;
			mainLeft = 0;
			mainHeight = 520;  // 13 rows times 40 pixels each
			mainWidth = 520;  // 13 columns times 40 pixels each
		}

		// set the variables for the player name(s) and grid size
		try
		{
			ArrayList playerNamesArrayList = new ArrayList();
			String playerNamesUntokenized = properties.getProperty("playernames", "");
			StringTokenizer playerNamesTokenizer = new StringTokenizer(playerNamesUntokenized, "|");
			while (playerNamesTokenizer.hasMoreTokens())
			{
				playerNamesArrayList.add(playerNamesTokenizer.nextToken());
			}
			Collections.sort(playerNamesArrayList);
			Object[] playerNamesObjectArray = playerNamesArrayList.toArray();
			int playerNamesObjectArrayLength = playerNamesObjectArray.length;
			playerNames = new String[playerNamesObjectArrayLength];
			for (int i = 0; i < playerNamesObjectArrayLength; i++)
			{
				playerNames[i] = (String)playerNamesObjectArray[i];
			}
			playerName = properties.getProperty("playername", "Anonymous");
			gridSize = Integer.parseInt(properties.getProperty("gridsize", "12"));
		}
		catch(Exception e)
		{
			playerNames = new String[] {"", "Anonymous"};
			playerName = "Anonymous";
			gridSize = 12;
		}
	}


	/**
	 * [Add javadoc comments here]
	 */
	private void run()
	{
		TTJSettingsDialog ttjSettingsDialog = new TTJSettingsDialog(this);
		ttjSettingsDialog.setVisible(true);
	}


    /**
     */
    private void loadProperties()
    {
		// get properties
		properties = new Properties();
		FileInputStream fis = null;

		try
		{
			System.out.println("Reading TimesTableJupe.properties");  //debug
			fis = new FileInputStream("TimesTableJupe.properties");
		}
		catch(FileNotFoundException e)
		{
			System.out.println("Could not find the TimesTableJupe.properties file for read.");	//debug
			return;
		}

		try
		{
			properties.load(fis);
			fis.close();
		}
		catch(IOException e)
		{
			System.out.println("Error reading TimesTableJupe.properties file: " + e.toString());	//debug
			return;
		}
    }


	/**
	 */
	protected void saveProperties()
	{
		try
		{
			properties.setProperty("settingstop", Integer.toString(settingsTop));
			properties.setProperty("settingsleft", Integer.toString(settingsLeft));
			properties.setProperty("settingsheight", Integer.toString(settingsHeight));
			properties.setProperty("settingswidth", Integer.toString(settingsWidth));
			properties.setProperty("resultstop", Integer.toString(resultsTop));
			properties.setProperty("resultsleft", Integer.toString(resultsLeft));
			properties.setProperty("resultsheight", Integer.toString(resultsHeight));
			properties.setProperty("resultswidth", Integer.toString(resultsWidth));
			properties.setProperty("maintop", Integer.toString(mainTop));
			properties.setProperty("mainleft", Integer.toString(mainLeft));
			properties.setProperty("mainheight", Integer.toString(mainHeight));
			properties.setProperty("mainwidth", Integer.toString(mainWidth));
			properties.setProperty("gridsize", Integer.toString(gridSize));
			properties.setProperty("playername", playerName);

			String playerNamesAllOneString = playerName;
			for (int i = 0; i < playerNames.length; i++)
			{
				if (playerName.equals(playerNames[i]))
				{
					// do not add the name, because it's already there
				}
				else
				{
					// the name is not already there, so go ahead and add it
					playerNamesAllOneString += "|" + playerNames[i];
				}
			}
			properties.setProperty("playernames", playerNamesAllOneString);
		}
		catch(Exception e)
		{
			properties.setProperty("settingstop", "0");
			properties.setProperty("settingsleft", "0");
			properties.setProperty("settingsheight", "120");
			properties.setProperty("settingswidth", "300");
			properties.setProperty("resultstop", "0");
			properties.setProperty("resultsleft", "0");
			properties.setProperty("resultsheight", "120");
			properties.setProperty("resultswidth", "300");
			properties.setProperty("maintop", "0");
			properties.setProperty("mainleft", "0");
			properties.setProperty("mainheight", "520");
			properties.setProperty("mainwidth", "520");
			properties.setProperty("gridsize", "12");
			properties.setProperty("playername", "Anonymous");
			properties.setProperty("playernames", "");
		}

		storeProperties();
	}


	/**
	 */
	private void storeProperties()
	{
		// save user properties
		FileOutputStream fos = null;

		try
		{
			System.out.println("Writing TimesTableJupe.properties");  //debug
			fos = new FileOutputStream("TimesTableJupe.properties");
		}
		catch(FileNotFoundException e)
		{
			System.out.println("Could not find the TimesTableJupe.properties file for write.");  //debug
			return;
		}

		try
		{
			properties.store(fos, "--- Times Table Jupe program settings ---");
			fos.close();
		}
		catch(IOException e)
		{
			System.out.println("Error writing to TimesTableJupe.properties file: " + e.toString());  //debug
			return;
		}
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected int getSettingsTop()
	{
		return settingsTop;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected void setSettingsTop(int newsettingstop)
	{
		settingsTop = newsettingstop;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected int getSettingsLeft()
	{
		return settingsLeft;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected void setSettingsLeft(int newsettingsleft)
	{
		settingsLeft = newsettingsleft;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected int getSettingsHeight()
	{
		return settingsHeight;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected void setSettingsHeight(int newsettingsheight)
	{
		settingsHeight = newsettingsheight;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected int getSettingsWidth()
	{
		return settingsWidth;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected void setSettingsWidth(int newsettingswidth)
	{
		settingsWidth = newsettingswidth;
	}


	/**
	 * Accessor for the results dialog.
	 */
	protected int getResultsTop()
	{
		return resultsTop;
	}


	/**
	 * Accessor for the results dialog.
	 */
	protected void setResultsTop(int newresultstop)
	{
		resultsTop = newresultstop;
	}


	/**
	 * Accessor for the results dialog.
	 */
	protected int getResultsLeft()
	{
		return resultsLeft;
	}


	/**
	 * Accessor for the results dialog.
	 */
	protected void setResultsLeft(int newresultsleft)
	{
		resultsLeft = newresultsleft;
	}


	/**
	 * Accessor for the results dialog.
	 */
	protected int getResultsHeight()
	{
		return resultsHeight;
	}


	/**
	 * Accessor for the results dialog.
	 */
	protected void setResultsHeight(int newresultsheight)
	{
		resultsHeight = newresultsheight;
	}


	/**
	 * Accessor for the results dialog.
	 */
	protected int getResultsWidth()
	{
		return resultsWidth;
	}


	/**
	 * Accessor for the results dialog.
	 */
	protected void setResultsWidth(int newresultswidth)
	{
		resultsWidth = newresultswidth;
	}


	/**
	 * Accessor for the main grid window.
	 */
	protected int getMainTop()
	{
		return mainTop;
	}


	/**
	 * Accessor for the main grid window.
	 */
	protected void setMainTop(int newmaintop)
	{
		mainTop = newmaintop;
	}


	/**
	 * Accessor for the main grid window.
	 */
	protected int getMainLeft()
	{
		return mainLeft;
	}


	/**
	 * Accessor for the main grid window.
	 */
	protected void setMainLeft(int newmainleft)
	{
		mainLeft = newmainleft;
	}


	/**
	 * Accessor for the main grid window.
	 */
	protected int getMainHeight()
	{
		return mainHeight;
	}


	/**
	 * Accessor for the main grid window.
	 */
	protected void setMainHeight(int newmainheight)
	{
		mainHeight = newmainheight;
	}


	/**
	 * Accessor for the main grid window.
	 */
	protected int getMainWidth()
	{
		return mainWidth;
	}


	/**
	 * Accessor for the main grid window.
	 */
	protected void setMainWidth(int newmainwidth)
	{
		mainWidth = newmainwidth;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected String getPlayerName()
	{
		return playerName;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected void setPlayerName(String pName)
	{
		playerName = pName;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected String[] getPlayerNames()
	{
		return playerNames;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected void setPlayerNames(String[] pNames)
	{
		playerNames = pNames;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected int getGridSize()
	{
		return gridSize;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected void setGridSize(int gSize)
	{
		gridSize = gSize;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected int getErrorCount()
	{
		return errorCount;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected void incrementErrorCount()
	{
		errorCount++;
	}


	/**
	 * Accessor for 'Reset' button event.
	 */
	protected void initializeErrorCount()
	{
		errorCount = 0;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected boolean getStartState()
	{
		return startState;
	}


	/**
	 * Accessor for the settings dialog.
	 */
	protected void setStartState(boolean state)
	{
		startState = state;
	}
}




/**
 * [Add javadoc comments here]
 */
class TTJTextField extends JTextField
{
	private int rownum;
	private int colnum;
	private int product;
	private int guess;
	private long guesstime;


	/**
	 * This constructor takes two arguments: the column number and the row number.
	 * @param row the grid row which this text field inhabits.
	 * @param col the grid column which this text field inhabits.
	 */
	public TTJTextField(int row, int col)
	{
		rownum = row;
		colnum = col;
		product = row * col;
		guess = 0;
		guesstime = 0L;
	}


	/**
	 * Returns the product.
	 * @return the product of rownum * colnum.
	 */
	public int getProduct()
	{
		return product;
	}


	/**
	 * Sets the guess.
	 * @param aGuess the guess.
	 */
	public void setGuess(int aGuess)
	{
		guess = aGuess;
	}


	/**
	 * Returns the guess.
	 * @return the guess.
	 */
	public int getGuess()
	{
		return guess;
	}


	/**
	 * Sets the time (in milliseconds) taken to enter the guess.
	 * @param aGuessTime the time (in milliseconds) taken to enter the guess.
	 */
	public void setGuessTime(long aGuessTime)
	{
		guesstime = aGuessTime;
	}


	/**
	 * Returns the time (in milliseconds) taken to enter the guess.
	 * @return the time (in milliseconds) taken to enter the guess.
	 */
	public long getGuessTime()
	{
		return guesstime;
	}
}




/**
 * [Add javadoc comments here]
 */
class TTJFrame extends JFrame
{
	private TimesTableJupe ttjupe;
	private Container contentPane;
	private TTJTextField[][] inputField;
	private JPanel gridPanel;
	private JPanel southPanel;
	private JPanel buttonPanel;
	private JButton startButton;
	private JButton resetButton;
	private JToggleButton pauseButton;
	private JButton exitButton;
	private JLabel timerLabel;
	private JMenuBar menuBar;
	private JMenu fileMenu;
	private JMenu viewMenu;
	private JMenu helpMenu;
	private JMenuItem settingsMenuItem;
	private JMenuItem resultsMenuItem;
	private JMenuItem exitMenuItem;
	private JMenuItem answersMenuItem;
	private JMenuItem answerTimesMenuItem;
	private JMenuItem helpMenuItem;
	private JMenuItem aboutMenuItem;
	private Random pairsIndexRandomGenerator;
	private ArrayList rowcolPairs;
	private int[] rowcolPair;
	private Timer timer;
	private final float DEFAULTLABELFONTSIZE = 12.0F;
	private final float SELECTEDLABELFONTSIZE = 24.0F;
	private final Color DEFAULTLABELFONTCOLOR = new Color(102, 102, 153);
	private final Color SELECTEDLABELFONTCOLOR = Color.black;
	private final Color CORRECTANSWERFONTCOLOR = new Color(0, 100, 0);  //dark green
	private int gridSize;
	private int gridPanelSize;
	private long elapsedTime;  // measured in milliseconds
	private long timeAtCurrentTimerAction;
	private long timeAtPreviousTimerAction;
	private long guessTimeStart;
	private long guessTimeEnd;
	private String playerName;
	private String revnumber = "$Revision: 92 $";
	private String revdate = "$Date: 2017-09-10 17:54:15 -0400 (Sun, 10 Sep 2017) $";
	private final String REVISIONNUMBER = revnumber.substring((revnumber.indexOf("$") + 1), revnumber.lastIndexOf(" $"));
	private final String REVISIONDATE = revdate.substring((revdate.indexOf("$") + 1), revdate.lastIndexOf(" $"));




	/**
	 * Constructor.
	 */
	public TTJFrame(TimesTableJupe parent)
	{
		super();
		ttjupe = parent;
		doInits();
	}


	// initializations
	private void doInits()
	{
		this.setVisible(false);

		menuBar = new JMenuBar();
		this.setJMenuBar(menuBar);

		fileMenu = new JMenu("File");
		fileMenu.setMnemonic(KeyEvent.VK_F);
		menuBar.add(fileMenu);

		viewMenu = new JMenu("View");
		menuBar.add(viewMenu);

		helpMenu = new JMenu("Help");
		menuBar.add(helpMenu);

		settingsMenuItem = new JMenuItem("Change Settings...");
		fileMenu.add(settingsMenuItem);
		settingsMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				settingsMenuItemActionPerformed(event);
			}
		});

		resultsMenuItem = new JMenuItem("Show Results...");
		fileMenu.add(resultsMenuItem);
		resultsMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				resultsMenuItemActionPerformed(event);
			}
		});

		fileMenu.add(new JSeparator());

		exitMenuItem = new JMenuItem("Exit");
		exitMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				exitMenuItemActionPerformed(event);
			}
		});
		exitMenuItem.setMnemonic(KeyEvent.VK_X);
		fileMenu.add(exitMenuItem);

		answersMenuItem = new JMenuItem("Answers");
		answersMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				answersMenuItemActionPerformed(event);
			}
		});
		answersMenuItem.setEnabled(false);
		viewMenu.add(answersMenuItem);

		answerTimesMenuItem = new JMenuItem("Answer Times");
		answerTimesMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				timesMenuItemActionPerformed(event);
			}
		});
		answerTimesMenuItem.setEnabled(false);
		viewMenu.add(answerTimesMenuItem);

		helpMenuItem = new JMenuItem("Help...");
		helpMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				helpMenuItemActionPerformed(event);
			}
		});
		helpMenu.add(helpMenuItem);

		aboutMenuItem = new JMenuItem("About TTJ...");
		aboutMenuItem.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				aboutMenuItemActionPerformed(event);
			}
		});
		helpMenu.add(aboutMenuItem);

		playerName = ttjupe.getPlayerName();
		gridSize = ttjupe.getGridSize();

		// set up the grid panel
		gridPanel = new JPanel();
		gridPanelSize = gridSize + 1;
		gridPanel.setLayout(new GridLayout(gridPanelSize, gridPanelSize));

		// set up the input fields
		inputField = new TTJTextField[gridSize][gridSize];
		for (int i = 0; i < gridSize; i++) //rownum
		{
			for (int j = 0; j < gridSize; j++) //colnum
			{
				inputField[i][j] = new TTJTextField(i + 1, j + 1);
				inputField[i][j].setHorizontalAlignment(TTJTextField.CENTER);
				inputField[i][j].setEditable(false);
				inputField[i][j].addKeyListener(new KeyAdapter()
				{
					public void keyPressed(KeyEvent event)
					{
						inputKeyPressed(event);
					}
				});
				inputField[i][j].addMouseListener(new MouseAdapter()
				{
					public void mouseEntered(MouseEvent event)
					{
						inputMouseEntered(event);
					}
					public void mouseExited(MouseEvent event)
					{
						inputMouseExited(event);
					}
				});
			}
		}

		for (int k = 0; k < gridPanelSize; k++) //rows
		{
			for (int l = 0; l < gridPanelSize; l++) //cols
			{
				if ((k == 0) && (l == 0))
				{
					gridPanel.add(new JLabel(""));
				}
				else if ((k == 0) && (l > 0))
				{
					gridPanel.add(new JLabel("" + l, SwingConstants.CENTER));
				}
				else if ((k > 0) && (l == 0))
				{
					gridPanel.add(new JLabel("" + k, SwingConstants.CENTER));
				}
				else
				{
					gridPanel.add(inputField[k - 1][l - 1]);
				}
			}
		}


		// set up the button panel, with JPanel's default flow layout

		startButton = new JButton();
		startButton.setText("Start");
		startButton.setEnabled(true);
		startButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				startButtonActionPerformed(event);
			}
		});

		pauseButton = new JToggleButton();
		pauseButton.setText("Pause");
		pauseButton.setEnabled(false);
		pauseButton.setSelected(false);
		pauseButton.addItemListener(new ItemListener()
		{
			public void itemStateChanged(ItemEvent event)
			{
				pauseButtonItemStateChanged(event);
			}
		});

		resetButton = new JButton();
		resetButton.setText("Reset");
		resetButton.setEnabled(false);
		resetButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				resetButtonActionPerformed(event);
			}
		});

		exitButton = new JButton();
		exitButton.setText("Exit");
		exitButton.setEnabled(true);
		exitButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				exitButtonActionPerformed(event);
			}
		});

		timeAtCurrentTimerAction = 0L;
		timeAtPreviousTimerAction = 0L;
		elapsedTime = 0L;

		timerLabel = new JLabel();
		displayElapsedTime();

		timer = new Timer(1000, new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				timerActionPerformed(event);
			}
		});

		buttonPanel = new JPanel();
		buttonPanel.add(startButton);
		buttonPanel.add(pauseButton);
		buttonPanel.add(resetButton);
		buttonPanel.add(exitButton);

		southPanel = new JPanel();
		southPanel.setLayout(new BorderLayout());
		southPanel.add(buttonPanel, BorderLayout.CENTER);
		southPanel.add(timerLabel, BorderLayout.EAST);

		contentPane = this.getContentPane();
		contentPane.add(gridPanel, BorderLayout.CENTER);
		contentPane.add(southPanel, BorderLayout.SOUTH);

		pairsIndexRandomGenerator = new Random();
		pairsIndexRandomGenerator.setSeed(System.currentTimeMillis());

		rowcolPairs = new ArrayList(gridSize * gridSize);

		for (int r = 0; r < gridSize; r++) //rows
		{
			for (int c = 0; c < gridSize; c++) //cols
			{
				rowcolPairs.add(new int[] {r, c});
			}
		}


		// event handler for window closing
		this.addWindowListener(new WindowAdapter()
		{
			public void windowClosing(WindowEvent event)
			{
				exitApp();
			}
		});

		// set main window's initial size and location on the screen
		int top = ttjupe.getMainTop();
		int left = ttjupe.getMainLeft();
		int height = ttjupe.getMainHeight();
		int width = ttjupe.getMainWidth();

		this.setTitle("Times Table Jupe");
		this.setLocation(left, top);
		this.setSize(width, height);
		this.setVisible(true);
	}



	private void selectNewInputField()
	{
		if (rowcolPair != null)
		{
			int[] oldRowColPair = rowcolPair;
			int oldRowNumber = oldRowColPair[0];
			int oldColNumber = oldRowColPair[1];

			int oldRowLabelIndex = (oldRowNumber + 1) * gridPanelSize;
			JLabel oldRowLabel = (JLabel)gridPanel.getComponent(oldRowLabelIndex);
			Font oldRowLabelFont = oldRowLabel.getFont();
			oldRowLabel.setFont(oldRowLabelFont.deriveFont(DEFAULTLABELFONTSIZE));
			oldRowLabel.setForeground(DEFAULTLABELFONTCOLOR);

			int oldColLabelIndex = oldColNumber + 1;
			JLabel oldColLabel = (JLabel)gridPanel.getComponent(oldColLabelIndex);
			Font oldColLabelFont = oldColLabel.getFont();
			oldColLabel.setFont(oldColLabelFont.deriveFont(DEFAULTLABELFONTSIZE));
			oldColLabel.setForeground(DEFAULTLABELFONTCOLOR);
		}

		int rowcolPairsSize = rowcolPairs.size();
		if (rowcolPairsSize > 0)
		{
			int pairsIndexRandomNumber = pairsIndexRandomGenerator.nextInt(rowcolPairsSize);
			rowcolPair = (int[])rowcolPairs.remove(pairsIndexRandomNumber);
			int rowNumber = rowcolPair[0];
			int colNumber = rowcolPair[1];
			inputField[rowNumber][colNumber].setEditable(true);
			inputField[rowNumber][colNumber].requestFocus();

			int rowLabelIndex = (rowNumber + 1) * gridPanelSize;
			JLabel rowLabel = (JLabel)gridPanel.getComponent(rowLabelIndex);
			Font rowLabelFont = rowLabel.getFont();
			rowLabel.setFont(rowLabelFont.deriveFont(SELECTEDLABELFONTSIZE));
			rowLabel.setForeground(SELECTEDLABELFONTCOLOR);

			int colLabelIndex = colNumber + 1;
			JLabel colLabel = (JLabel)gridPanel.getComponent(colLabelIndex);
			Font colLabelFont = colLabel.getFont();
			colLabel.setFont(colLabelFont.deriveFont(SELECTEDLABELFONTSIZE));
			colLabel.setForeground(SELECTEDLABELFONTCOLOR);

			guessTimeStart = new Date().getTime();
		}
		else
		{
			timer.stop();
			pauseButton.setEnabled(false);
			answersMenuItem.setEnabled(false);
			answerTimesMenuItem.setEnabled(true);
			Toolkit.getDefaultToolkit().beep(); //debug
			saveResult();
			showResultMessage();
		}
	}



	private void saveResult()
	{
		try
		{
			long currentTime = new Date().getTime();
			int elapsedSeconds = (int)(elapsedTime/1000L);
			BufferedWriter writer = new BufferedWriter(new FileWriter("TimesTableJupe.results", true));
			String result = playerName + "\t" + gridSize + "\t" + ttjupe.getErrorCount() + "\t" + elapsedSeconds + "\t" + currentTime + "\n";
			writer.write(result, 0, result.length());
			writer.flush();
			writer.close();
		}
		catch(Exception e)
		{
			e.printStackTrace();
		}
	}



	private void showResultMessage()
	{
		try
		{
			/*
			 * Note to self:
			 * 1. Read in each line of the results file. If the line contains the corresponding playername, gridsize, and errorcount, create a TTJResult object for that row.
			 * 2. Add the object to an ArrayList.
			 * 3. Use Collections.sort(List, Comparator) to sort the ArrayList on elapsedSeconds.
			 * 4. Find out where this result is in the ArrayList.
			 */

			int elapsedSeconds = (int)(elapsedTime/1000L);
			ArrayList resultList = new ArrayList();
			BufferedReader reader = new BufferedReader(new FileReader("TimesTableJupe.results"));
			String resultLine = reader.readLine();
			while (resultLine != null)
			{
				StringTokenizer resultTokenizer = new StringTokenizer(resultLine, "\t");
				while (resultTokenizer.hasMoreTokens())
				{
					String playername = resultTokenizer.nextToken();                   // player name
					int gridsize = Integer.parseInt(resultTokenizer.nextToken());      // grid size
					int errorcount = Integer.parseInt(resultTokenizer.nextToken());    // error count
					int secondscount = Integer.parseInt(resultTokenizer.nextToken());  // elapsed seconds
					long resultdate = Long.parseLong(resultTokenizer.nextToken());     // the date of the result (in milliseconds)

					if ((playername.equals(ttjupe.getPlayerName())) && (gridsize == ttjupe.getGridSize()) && (errorcount == ttjupe.getErrorCount()))
					{
						TTJResult result = new TTJResult(playername, gridsize, errorcount, secondscount, resultdate);
						resultList.add(result);
					}
				}
				resultLine = reader.readLine();
			}
			reader.close();
			Collections.sort(resultList, new SecondsCountComparator());

			// find out where the current result is in the list
			int resultListSize = resultList.size();
			int resultRank = 0;
			while (resultRank == 0)
			{
				for (int listPosition = 0; listPosition < resultListSize; listPosition++)
				{
					TTJResult result = (TTJResult)resultList.get(listPosition);
					int secondscount = result.getSecondsCount();
					if (secondscount == elapsedSeconds)
					{
						resultRank = listPosition + 1;
						break;
					}
				}
			}

			String message = "Nice going, " + playerName + "!\n\n";
			message += "You finished the " + gridSize + " times table in a time of " + getElapsedTimeAsString() + " with " + ttjupe.getErrorCount() + " mistakes.\n\n";
			message += playerName + ", of all your results for the " + gridSize + " times table with " + ttjupe.getErrorCount() + " mistakes, this result ranks as number " + resultRank + " out of " + resultListSize + ".\n";
			JOptionPane.showMessageDialog(this, message, "Finished!", JOptionPane.INFORMATION_MESSAGE);
		}
		catch(Exception e)
		{
			e.printStackTrace();
		}
	}




	private void timerActionPerformed(ActionEvent event)
	{
		timeAtCurrentTimerAction = new Date().getTime();
		elapsedTime += (timeAtCurrentTimerAction - timeAtPreviousTimerAction);
		displayElapsedTime();
		timeAtPreviousTimerAction = timeAtCurrentTimerAction;
	}




	private void displayElapsedTime()
	{
		//System.out.println("Nuts!");  //debug
		timerLabel.setText(getElapsedTimeAsString() + "    ");
	}




	private String getElapsedTimeAsString()
	{
		int elapsedSeconds = (int)(elapsedTime/1000L);
		//System.out.println("'elapsedTime' is: " + elapsedTime);  //debug
		//System.out.println("'elapsedSeconds' is: " + elapsedSeconds);  //debug
		int minutes = (elapsedSeconds / 60);
		int seconds = (elapsedSeconds % 60);
		NumberFormat formatter = NumberFormat.getNumberInstance();
		formatter.setMinimumIntegerDigits(2);
		String elapsedTimeAsString = minutes + ":" + formatter.format(seconds);
		return elapsedTimeAsString;
	}



	private void startButtonActionPerformed(ActionEvent event)
	{
		startButton.setEnabled(false);
		resetButton.setEnabled(true);
		pauseButton.setEnabled(true);
		settingsMenuItem.setEnabled(false);
		selectNewInputField();

		timeAtCurrentTimerAction = new Date().getTime();
		timeAtPreviousTimerAction = new Date().getTime();
		timer.start();
	}


	private void pauseButtonItemStateChanged(ItemEvent event)
	{
		//System.out.println("Executing the pauseButtonItemStateChanged() method.");  //debug
		if (pauseButton.isSelected() == true)
		{
			pauseButton.setText("Resume");
			rowcolPairs.add(rowcolPair);
			int rowNumber = rowcolPair[0];
			int colNumber = rowcolPair[1];
			inputField[rowNumber][colNumber].setText("");
			inputField[rowNumber][colNumber].setEditable(false);

			int rowLabelIndex = (rowNumber + 1) * gridPanelSize;
			JLabel rowLabel = (JLabel)gridPanel.getComponent(rowLabelIndex);
			Font rowLabelFont = rowLabel.getFont();
			rowLabel.setFont(rowLabelFont.deriveFont(DEFAULTLABELFONTSIZE));
			rowLabel.setForeground(DEFAULTLABELFONTCOLOR);

			int colLabelIndex = colNumber + 1;
			JLabel colLabel = (JLabel)gridPanel.getComponent(colLabelIndex);
			Font colLabelFont = colLabel.getFont();
			colLabel.setFont(colLabelFont.deriveFont(DEFAULTLABELFONTSIZE));
			colLabel.setForeground(DEFAULTLABELFONTCOLOR);

			timeAtCurrentTimerAction = 0L;
			timeAtPreviousTimerAction = 0L;
			timer.stop();
		}
		else if ((pauseButton.isSelected() == false) && (startButton.isEnabled() == false))
		{
			pauseButton.setText("Pause");
			selectNewInputField();
			timeAtCurrentTimerAction = new Date().getTime();
			timeAtPreviousTimerAction = new Date().getTime();
			timer.start();
		}
		else
		{
			pauseButton.setText("Pause");
			timeAtCurrentTimerAction = new Date().getTime();
			timeAtPreviousTimerAction = new Date().getTime();
			timer.start();
		}
	}


	private void resetButtonActionPerformed(ActionEvent event)
	{
		for (int i = 0; i < gridSize; i++) //rownum
		{
			for (int j = 0; j < gridSize; j++) //colnum
			{
				inputField[i][j].setText("");
				inputField[i][j].setForeground(Color.black);
				inputField[i][j].setEditable(false);
			}
		}

		if (rowcolPair != null)
		{
			int rowNumber = rowcolPair[0];
			int colNumber = rowcolPair[1];

			int rowLabelIndex = (rowNumber + 1) * gridPanelSize;
			JLabel rowLabel = (JLabel)gridPanel.getComponent(rowLabelIndex);
			Font rowLabelFont = rowLabel.getFont();
			rowLabel.setFont(rowLabelFont.deriveFont(DEFAULTLABELFONTSIZE));
			rowLabel.setForeground(DEFAULTLABELFONTCOLOR);

			int colLabelIndex = colNumber + 1;
			JLabel colLabel = (JLabel)gridPanel.getComponent(colLabelIndex);
			Font colLabelFont = colLabel.getFont();
			colLabel.setFont(colLabelFont.deriveFont(DEFAULTLABELFONTSIZE));
			colLabel.setForeground(DEFAULTLABELFONTCOLOR);
		}

		rowcolPairs = new ArrayList(gridSize * gridSize);

		for (int r = 0; r < gridSize; r++) //rows
		{
			for (int c = 0; c < gridSize; c++) //cols
			{
				rowcolPairs.add(new int[] {r, c});
			}
		}

		startButton.setEnabled(true);
		resetButton.setEnabled(false);
		pauseButton.setSelected(false);
		pauseButton.setText("Pause");
		pauseButton.setEnabled(false);
		settingsMenuItem.setEnabled(true);
		answersMenuItem.setEnabled(false);
		answerTimesMenuItem.setEnabled(false);

		ttjupe.initializeErrorCount();

		timer.stop();
		timeAtCurrentTimerAction = 0L;
		timeAtPreviousTimerAction = 0L;
		elapsedTime = 0L;
		displayElapsedTime();
	}




	private void inputKeyPressed(KeyEvent event)
	{
		TTJTextField thisInputField = (TTJTextField)event.getSource();

		if (event.getKeyCode() == event.VK_ENTER)
		{
			try
			{
				int thisProduct = thisInputField.getProduct();
				int thisGuess = Integer.parseInt(thisInputField.getText());
				thisInputField.setGuess(thisGuess);
				guessTimeEnd = new Date().getTime();
				thisInputField.setGuessTime(guessTimeEnd - guessTimeStart);

				if (thisGuess == thisProduct)  //correct answer
				{
					thisInputField.setForeground(Color.black);
					thisInputField.setEditable(false);
					selectNewInputField();
				}
				else  //incorrect answer
				{
					thisInputField.setForeground(Color.red);
					Toolkit.getDefaultToolkit().beep(); //debug
					thisInputField.setEditable(false);
					ttjupe.incrementErrorCount();
					selectNewInputField();
				}
			}
			catch (NumberFormatException exception)  //invalid answer (not a number)
			{
				thisInputField.setText("");
				Toolkit.getDefaultToolkit().beep(); //debug
			}
			catch (Exception exception)
			{
				thisInputField.setText("");
				Toolkit.getDefaultToolkit().beep(); //debug
			}
		}
	}




	private void inputMouseEntered(MouseEvent event)
	{
		int rowcolPairsSize = rowcolPairs.size();
		if (rowcolPairsSize == 0)
		{
			TTJTextField thisInputField = (TTJTextField)event.getSource();
			int thisProduct = thisInputField.getProduct();
			int thisGuess = thisInputField.getGuess();
			String thisText = thisInputField.getText();

			if ((thisText != null) && (thisText.length() > 0))
			{
				if (thisGuess != thisProduct)
				{
					thisInputField.setText("" + thisProduct);
					thisInputField.setForeground(CORRECTANSWERFONTCOLOR);
				}
			}
		}
	}




	private void inputMouseExited(MouseEvent event)
	{
		int rowcolPairsSize = rowcolPairs.size();
		if (rowcolPairsSize == 0)
		{
			TTJTextField thisInputField = (TTJTextField)event.getSource();
			int thisProduct = thisInputField.getProduct();
			int thisGuess = thisInputField.getGuess();
			String thisText = thisInputField.getText();

			if ((thisText != null) && (thisText.length() > 0))
			{
				if (thisGuess != thisProduct)
				{
					thisInputField.setText("" + thisGuess);
					thisInputField.setForeground(Color.red);
				}
			}
		}
	}




	private void settingsMenuItemActionPerformed(ActionEvent event)
	{
		TTJSettingsDialog ttjSettingsDialog = new TTJSettingsDialog(ttjupe);
		ttjSettingsDialog.setVisible(true);
		exitCleanup();
		this.setVisible(false);
		this.dispose();
	}




	private void resultsMenuItemActionPerformed(ActionEvent event)
	{
		TTJResultsDialog ttjResultsDialog = new TTJResultsDialog(ttjupe);
		ttjResultsDialog.setVisible(true);
	}




	private void answersMenuItemActionPerformed(ActionEvent event)
	{
		for (int r = 0; r < gridSize; r++) //rownum
		{
			for (int c = 0; c < gridSize; c++) //colnum
			{
				int guess = inputField[r][c].getGuess();
				String guessString = Integer.toString(guess);
				inputField[r][c].setText(guessString);
			}
		}
		answersMenuItem.setEnabled(false);
		answerTimesMenuItem.setEnabled(true);
	}




	private void timesMenuItemActionPerformed(ActionEvent event)
	{
		//long myTotalTime = 0; //debug

		for (int r = 0; r < gridSize; r++) //rownum
		{
			for (int c = 0; c < gridSize; c++) //colnum
			{
				long guessTime = inputField[r][c].getGuessTime();                    // 3789
				//myTotalTime += guessTime; //debug
				//System.out.println("guessTime is: " + guessTime); //debug
				double guessTimeDivided = guessTime/100.0;                           // 37.89
				//System.out.println("guessTimeDivided is: " + guessTimeDivided); //debug
				int guessTimeRounded = (int)Math.round(guessTimeDivided);            // 38
				//System.out.println("guessTimeRounded is: " + guessTimeRounded); //debug
				double guessTimeAsSeconds = guessTimeRounded/10.0;                   // 3.8
				//System.out.println("guessTimeAsSeconds is: " + guessTimeAsSeconds + "\n\n"); //debug
				String guessTimeString = Double.toString(guessTimeAsSeconds);
				inputField[r][c].setText(guessTimeString);
			}
		}
		answersMenuItem.setEnabled(true);
		answerTimesMenuItem.setEnabled(false);
		//System.out.println("myTotalTime is: " + myTotalTime); //debug
	}




	private void helpMenuItemActionPerformed(ActionEvent event)
	{
	}




	private void aboutMenuItemActionPerformed(ActionEvent event)
	{
		String message = "Times Table Jupe\n";
		message += "Copyright (c) Stephen Jupe\n";
		message += REVISIONNUMBER + "\n";
		message += REVISIONDATE + "\n";
		JOptionPane.showMessageDialog(this, message, "About", JOptionPane.INFORMATION_MESSAGE);
	}




	private void exitMenuItemActionPerformed(ActionEvent event)
	{
		//System.out.println("Executing the exitMenuItemActionPerformed() method.");  //debug
		exitApp();
	}




	private void exitButtonActionPerformed(ActionEvent event)
	{
		//System.out.println("Executing the exitButtonActionPerformed() method.");  //debug
		exitApp();
	}




	private void exitCleanup()
	{
		//System.out.println("Executing the exitCleanup() method.");  //debug
		timer.stop();

		// save main window's size and location on the screen
		Point mainPoint = this.getLocationOnScreen();
		Dimension mainDimension = this.getSize();
		ttjupe.setMainTop(mainPoint.y);
		ttjupe.setMainLeft(mainPoint.x);
		ttjupe.setMainHeight(mainDimension.height);
		ttjupe.setMainWidth(mainDimension.width);
	}


	private void exitApp()
	{
		//System.out.println("Executing the exitApp() method.");  //debug
		exitCleanup();
		ttjupe.saveProperties();
		this.dispose();
		System.exit(0);
	}
}




/**
 * [Add javadoc comments here]
 */
class TTJSettingsDialog extends JDialog
{
	private TimesTableJupe ttjupe;
	private Container contentPane;
	private JPanel playerNamePanel;
	private JLabel playerNameLabel;
	private JComboBox playerNameComboBox;
	private JPanel gridSizePanel;
	private JLabel gridSizeLabel;
	private JComboBox gridSizeComboBox;
	private JPanel buttonPanel;
	private JButton okayButton;
	private JButton cancelButton;
	private JButton exitButton;
	private JButton helpButton;
	private String[] playerNames;
	private String playerName;
	private String gridSize;
	private boolean startState;


	/**
	 * Constructor.
	 */
	public TTJSettingsDialog(TimesTableJupe parent)
	{
		super();
		ttjupe = parent;
		doInits();
	}


	private void doInits()
	{
		playerNames = ttjupe.getPlayerNames();
		playerName = ttjupe.getPlayerName();
		gridSize = Integer.toString(ttjupe.getGridSize());

		playerNameLabel = new JLabel();
		playerNameLabel.setText("Player Name: ");

		playerNameComboBox = new JComboBox(playerNames);
		playerNameComboBox.setSelectedItem(playerName);
		playerNameComboBox.setEditable(true);
		playerNameComboBox.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				playerName = (String)playerNameComboBox.getSelectedItem();
			}
		});

		playerNamePanel = new JPanel();
		playerNamePanel.setLayout(new BoxLayout(playerNamePanel, BoxLayout.X_AXIS));
		playerNamePanel.add(playerNameLabel);
		playerNamePanel.add(playerNameComboBox);

		gridSizeLabel = new JLabel();
		gridSizeLabel.setText("Grid Size: ");

		gridSizeComboBox = new JComboBox(new String[] {"5", "6", "7", "8", "9", "10", "11", "12"});
		gridSizeComboBox.setSelectedItem(gridSize);
		gridSizeComboBox.addItemListener(new ItemListener()
		{
			public void itemStateChanged(ItemEvent event)
			{
				gridSize = (String)gridSizeComboBox.getSelectedItem();
			}
		});

		gridSizePanel = new JPanel();
		gridSizePanel.setLayout(new BoxLayout(gridSizePanel, BoxLayout.X_AXIS));
		gridSizePanel.add(gridSizeLabel);
		gridSizePanel.add(gridSizeComboBox);

		okayButton = new JButton();
		okayButton.setText("OK");
		okayButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				okayButtonActionPerformed(event);
			}
		});

		cancelButton = new JButton();
		cancelButton.setText("Cancel");
		cancelButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				cancelButtonActionPerformed(event);
			}
		});
		startState = ttjupe.getStartState();
		if (startState)
		{
			cancelButton.setEnabled(false);
			ttjupe.setStartState(false);
		}

		exitButton = new JButton();
		exitButton.setText("Exit");
		exitButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				exitButtonActionPerformed(event);
			}
		});

		helpButton = new JButton();
		helpButton.setText("Help");

		buttonPanel = new JPanel();
		buttonPanel.add(okayButton);
		buttonPanel.add(cancelButton);
		buttonPanel.add(exitButton);
		buttonPanel.add(helpButton);

		contentPane = this.getContentPane();
		contentPane.add(playerNamePanel, BorderLayout.NORTH);
		contentPane.add(gridSizePanel, BorderLayout.CENTER);
		contentPane.add(buttonPanel, BorderLayout.SOUTH);


		// set the settings dialog's initial size and location on the screen
		int top = ttjupe.getSettingsTop();
		int left = ttjupe.getSettingsLeft();
		int height = ttjupe.getSettingsHeight();
		int width = ttjupe.getSettingsWidth();

		this.setTitle("TTJ Settings");
		this.setLocation(left, top);
		this.setSize(width, height);
		this.setVisible(true);
		this.setModal(true);
		this.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

		// event handler for window closing
		this.addWindowListener(new WindowAdapter()
		{
			public void windowClosing(WindowEvent event)
			{
				closeDialog(false);
			}
		});
	}


	private void okayButtonActionPerformed(ActionEvent event)
	{
		closeDialog(true);
	}


	private void cancelButtonActionPerformed(ActionEvent event)
	{
		closeDialog(false);
	}


	private void exitButtonActionPerformed(ActionEvent event)
	{
		ttjupe.saveProperties();
		this.setVisible(false);
		this.dispose();
		System.exit(0);
	}


	private void closeDialog(boolean newSettings)
	{
		// save the settings dialog's size and location on the screen
		Point settingsPoint = this.getLocationOnScreen();
		Dimension settingsDimension = this.getSize();

		if (newSettings)  // player name or grid size or both have changed
		{
			boolean addName = true;
			for (int i = 0; i < playerNames.length; i++)
			{
				if (playerName.equals(playerNames[i]))
				{
					addName = false;
					break;
				}
			}
			if (addName)
			{
				String[] newPlayerNames = new String[playerNames.length + 1];
				for (int j = 0; j < playerNames.length; j++)
				{
					newPlayerNames[j] = playerNames[j];
				}
				newPlayerNames[newPlayerNames.length -1] = playerName;
				ttjupe.setPlayerNames(newPlayerNames);
			}
			ttjupe.setPlayerName(playerName);
			ttjupe.setGridSize(Integer.parseInt(gridSize));
		}
		else  // cancel button
		{
			// so use same settings
		}

		ttjupe.setSettingsTop(settingsPoint.y);
		ttjupe.setSettingsLeft(settingsPoint.x);
		ttjupe.setSettingsHeight(settingsDimension.height);
		ttjupe.setSettingsWidth(settingsDimension.width);
		TTJFrame ttjFrame = new TTJFrame(ttjupe);
		this.setVisible(false);
		this.dispose();
	}
}




/**
 * [Add javadoc comments here]
 */
class TTJResultsDialog extends JDialog
{
	private TimesTableJupe ttjupe;
	private Container contentPane;
	private JScrollPane scrollPane;
	private JPanel resultsTablePanel;
	private JTable resultsTable;
	private TTJResultsTableModel resultsTableModel;
	private JPanel optionPanel;
	private JPanel buttonPanel;
	private JButton helpButton;
	private JButton closeButton;


	/**
	 * Constructor.
	 */
	public TTJResultsDialog(TimesTableJupe parent)
	{
		super();
		ttjupe = parent;
		doInits();
	}


	private void doInits()
	{
		// set up the results table panel
		resultsTableModel = new TTJResultsTableModel();
		resultsTable = new JTable(resultsTableModel);

		DefaultTableCellRenderer renderer = new DefaultTableCellRenderer();
		renderer.setHorizontalAlignment(JLabel.CENTER);
		TableColumn tablecolumn = resultsTable.getColumn("Player Name");
		tablecolumn.setCellRenderer(renderer);
		tablecolumn = resultsTable.getColumn("Grid Size");
		tablecolumn.setCellRenderer(renderer);
		tablecolumn = resultsTable.getColumn("Errors");
		tablecolumn.setCellRenderer(renderer);
		tablecolumn = resultsTable.getColumn("Time");
		tablecolumn.setCellRenderer(renderer);
		tablecolumn = resultsTable.getColumn("Date");
		tablecolumn.setCellRenderer(renderer);

		scrollPane = new JScrollPane(resultsTable);

		resultsTablePanel = new JPanel();
		resultsTablePanel.setLayout(new BorderLayout());
		resultsTablePanel.add(scrollPane, BorderLayout.CENTER);

		closeButton = new JButton();
		closeButton.setText("Close");
		closeButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				closeButtonActionPerformed(event);
			}
		});

		helpButton = new JButton();
		helpButton.setText("Help");
		/*
		helpButton.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent event)
			{
				helpButtonActionPerformed(event);
			}
		});
		*/

		optionPanel = new JPanel();
		optionPanel.add(new JButton("Blah"));  // temporary placeholder

		buttonPanel = new JPanel();
		buttonPanel.add(closeButton);
		buttonPanel.add(helpButton);

		contentPane = this.getContentPane();
		contentPane.add(optionPanel, BorderLayout.NORTH);
		contentPane.add(resultsTablePanel, BorderLayout.CENTER);
		contentPane.add(buttonPanel, BorderLayout.SOUTH);

		// set the settings dialog's initial size and location on the screen
		int top = ttjupe.getResultsTop();
		int left = ttjupe.getResultsLeft();
		int height = ttjupe.getResultsHeight();
		int width = ttjupe.getResultsWidth();

		this.setTitle("TTJ Results");
		this.setLocation(left, top);
		this.setSize(width, height);
		this.setVisible(true);
		this.setModal(true);
		this.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

		// event handler for window closing
		this.addWindowListener(new WindowAdapter()
		{
			public void windowClosing(WindowEvent event)
			{
				closeDialog();
			}
		});
	}


	private void closeButtonActionPerformed(ActionEvent event)
	{
		closeDialog();
	}


	private void closeDialog()
	{
		// save the results dialog's size and location on the screen
		Point resultsPoint = this.getLocationOnScreen();
		Dimension resultsDimension = this.getSize();

		ttjupe.setResultsTop(resultsPoint.y);
		ttjupe.setResultsLeft(resultsPoint.x);
		ttjupe.setResultsHeight(resultsDimension.height);
		ttjupe.setResultsWidth(resultsDimension.width);

		this.setVisible(false);
		this.dispose();
	}
}




/**
 * [Add javadoc comments here]
 */
class TTJResult
{
	private String playerName;
	private int gridSize;
	private int errorCount;
	private int secondsCount;
	private long resultDate;


	/**
	 * This constructor takes five arguments: the player name, grid size, error count, elapsed seconds count, and result date.
	 * @param playername the name of the player.
	 * @param gridsize the size of the grid.
	 * @param errorcount the number of errors in the result.
	 * @param secondscount the number of elapsed seconds recorded for the result.
	 * @param resultdate the date (in milliseconds) on which the result was recorded.
	 */
	public TTJResult(String playername, int gridsize, int errorcount, int secondscount, long resultdate)
	{
		playerName = playername;
		gridSize = gridsize;
		errorCount = errorcount;
		secondsCount = secondscount;
		resultDate = resultdate;
	}


	/**
	 * Returns the player name.
	 * @return the player name.
	 */
	protected String getPlayerName()
	{
		return playerName;
	}


	/**
	 * Returns the grid size.
	 * @return the grid size.
	 */
	protected int getGridSize()
	{
		return gridSize;
	}


	/**
	 * Returns the error count.
	 * @return the error count.
	 */
	protected int getErrorCount()
	{
		return errorCount;
	}


	/**
	 * Returns the seconds count.
	 * @return the seconds count.
	 */
	protected int getSecondsCount()
	{
		return secondsCount;
	}


	/**
	 * Returns the result date.
	 * @return the result date.
	 */
	protected long getResultDate()
	{
		return resultDate;
	}
}




/**
 * Compares two TTJResult objects for sorting on the player name.
 */
class PlayerNameComparator implements Comparator
{
	public int compare(Object object1, Object object2)
	{
		TTJResult result1 = (TTJResult)object1;
		TTJResult result2 = (TTJResult)object2;

		String playerName1 = result1.getPlayerName();
		String playerName2 = result2.getPlayerName();

		return playerName1.compareTo(playerName2);
	}
}




/**
 * Compares two TTJResult objects for sorting on the grid size.
 */
class GridSizeComparator implements Comparator
{
	public int compare(Object object1, Object object2)
	{
		TTJResult result1 = (TTJResult)object1;
		TTJResult result2 = (TTJResult)object2;

		int gridSize1 = result1.getGridSize();
		int gridSize2 = result2.getGridSize();
		Integer gridSizeObject1 = new Integer(gridSize1);
		Integer gridSizeObject2 = new Integer(gridSize2);

		return gridSizeObject1.compareTo(gridSizeObject2);
	}
}




/**
 * Compares two TTJResult objects for sorting on the error count.
 */
class ErrorCountComparator implements Comparator
{
	public int compare(Object object1, Object object2)
	{
		TTJResult result1 = (TTJResult)object1;
		TTJResult result2 = (TTJResult)object2;

		int errorCount1 = result1.getErrorCount();
		int errorCount2 = result2.getErrorCount();
		Integer errorCountObject1 = new Integer(errorCount1);
		Integer errorCountObject2 = new Integer(errorCount2);

		return errorCountObject1.compareTo(errorCountObject2);
	}
}




/**
 * Compares two TTJResult objects for sorting on the seconds count.
 */
class SecondsCountComparator implements Comparator
{
	public int compare(Object object1, Object object2)
	{
		TTJResult result1 = (TTJResult)object1;
		TTJResult result2 = (TTJResult)object2;

		int secondsCount1 = result1.getSecondsCount();
		int secondsCount2 = result2.getSecondsCount();
		Integer secondsCountObject1 = new Integer(secondsCount1);
		Integer secondsCountObject2 = new Integer(secondsCount2);

		return secondsCountObject1.compareTo(secondsCountObject2);
	}
}




/**
 * Compares two TTJResult objects for sorting on the result date.
 */
class ResultDateComparator implements Comparator
{
	public int compare(Object object1, Object object2)
	{
		TTJResult result1 = (TTJResult)object1;
		TTJResult result2 = (TTJResult)object2;

		long resultDate1 = result1.getResultDate();
		long resultDate2 = result2.getResultDate();
		Long resultDateObject1 = new Long(resultDate1);
		Long resultDateObject2 = new Long(resultDate2);

		return resultDateObject1.compareTo(resultDateObject2);
	}
}




/**
 * [Add javadoc comments here]
 */
class TTJResultsTableModel extends AbstractTableModel
{
	private String[] columnNames = {"Player Name", "Grid Size", "Errors", "Time", "Date"};
	private ArrayList results;

	public TTJResultsTableModel()
	{
		results = new ArrayList();
		this.getResultsFromFile();
	}

	private void getResultsFromFile()
	{
		try
		{
			BufferedReader reader = new BufferedReader(new FileReader("TimesTableJupe.results"));
			String resultLine = reader.readLine();
			while (resultLine != null)
			{
				ArrayList result = new ArrayList();
				StringTokenizer resultTokenizer = new StringTokenizer(resultLine, "\t");
				while (resultTokenizer.hasMoreTokens())
				{
					result.add(resultTokenizer.nextToken());               // player name
					result.add(new Integer(resultTokenizer.nextToken()));  // grid size
					result.add(new Integer(resultTokenizer.nextToken()));  // error count
					result.add(new Integer(resultTokenizer.nextToken()));  // elapsed seconds
					result.add(new Date(Long.parseLong(resultTokenizer.nextToken())));  // the date of the result
				}
				results.add(result);
				resultLine = reader.readLine();
			}
			reader.close();
		}
		catch(Exception e)
		{
			e.printStackTrace();
		}
	}

	public int getColumnCount()
	{
		return columnNames.length;
	}

	public int getRowCount()
	{
		return results.size();
	}

	public String getColumnName(int column)
	{
		return columnNames[column];
	}

	public Object getValueAt(int row, int column)
	{
		ArrayList result = null;
		Object resultValue = null;

		try
		{
			result = (ArrayList)results.get(row);
			resultValue = result.get(column);
		}
		catch(Exception e)
		{
			e.printStackTrace();
		}

		return resultValue;
	}

	public Class getColumnClass(int column)
	{
		return getValueAt(0, column).getClass();
	}
}
